<?php

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 *
 * Create your own ghostpool_theme_setup() function to override in a child theme.
 *
 */
if ( ! function_exists( 'ghostpool_theme_setup' ) ) {
	function ghostpool_theme_setup() {

		// Localisation
		load_theme_textdomain( 'aardvark', trailingslashit( WP_LANG_DIR ) . 'themes/' );
		load_theme_textdomain( 'aardvark', get_theme_file_path( '/languages' ) );

		// Background customizer
		add_theme_support( 'custom-background' );

		// Visual/Gutenberg editor
		add_theme_support( 'wp-block-styles' );
		add_theme_support( 'align-wide' );
		add_theme_support( 'editor-styles' );
		add_editor_style( 'lib/framework/settings/assets/editor.css' );

		// Add default posts and comments RSS feed links to <head>
		add_theme_support( 'automatic-feed-links' );
		
		// Post formats
		add_theme_support( 'post-formats', array( 'aside', 'audio', 'chat', 'gallery', 'image', 'link', 'quote', 'status', 'video' ) );

		// Title support
		add_theme_support( 'title-tag' );
		
		// Indicate widget sidebars can use selective refresh in the Customizer
		add_theme_support( 'customize-selective-refresh-widgets' );
		
		// WooCommerce support
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-zoom' );
		add_theme_support( 'wc-product-gallery-lightbox' );
		add_theme_support( 'wc-product-gallery-slider' );		
		
		// Sensei support	
		add_theme_support( 'sensei' );
		
		// BuddyPress legacy support
		add_theme_support( 'buddypress-use-legacy' );
	
		// Register navigation menus
		register_nav_menus( array(
			'gp-main-header-primary-nav' => esc_html__( 'Main Header Primary Navigation', 'aardvark' ),
			'gp-mobile-primary-nav' => esc_html__( 'Mobile Primary Navigation', 'aardvark' ),
			'gp-main-header-secondary-nav' => esc_html__( 'Main Header Secondary Navigation', 'aardvark' ),
			'gp-top-header-left-nav' => esc_html__( 'Top Header Left Navigation', 'aardvark' ),
			'gp-top-header-right-nav' => esc_html__( 'Top Header Right Navigation', 'aardvark' ),
			'gp-profile-nav' => esc_html__( 'Profile Navigation', 'aardvark' ),
			'gp-mobile-profile-nav' => esc_html__( 'Mobile Profile Navigation', 'aardvark' ),
			'gp-footer-nav' => esc_html__( 'Footer Navigation', 'aardvark' ),
			'gp-side-menu-nav' => esc_html__( 'Side Menu Navigation', 'aardvark' ),
		) );

		// Disable LayerSlider activation features
		if ( function_exists( 'layerslider_set_as_theme' ) ) {
			layerslider_set_as_theme();
		}

		// Disable WPBakery redirection
		remove_action( 'admin_init', 'vc_page_welcome_redirect' );
		
				
	}
}
add_action( 'after_setup_theme', 'ghostpool_theme_setup' );
				
// Load Sensei functions
if ( function_exists( 'is_sensei' ) ) {
	require_once( get_theme_file_path( '/lib/inc/sensei-functions.php' ) );
}
		
/**
* Load framework
*
*/
if ( ! function_exists( 'ghostpool_load_framework' ) ) {
	function ghostpool_load_framework() {
		require_once( get_theme_file_path( '/lib/framework/framework.php' ) );						
	}
}
add_action( 'after_setup_theme', 'ghostpool_load_framework', 1 );

/**
* Load theme functions
*
*/
if ( ! function_exists( 'ghostpool_load_theme_functions' ) ) {
	function ghostpool_load_theme_functions() {

		// Load pagination functions
		require_once( get_theme_file_path( '/lib/inc/pagination-functions.php' ) );
				
		// Load query functions
		require_once( get_theme_file_path( '/lib/inc/query-functions.php' ) );

		// Load loop functions
		require_once( get_theme_file_path( '/lib/inc/loop-functions.php' ) );
		
		// Load filter functions		
		require_once( get_theme_file_path( '/lib/inc/filter-functions.php' ) );
						
		// Load custom menu walker
		require_once( get_theme_file_path( '/lib/menus/custom-menu-walker.php' ) );

		// Load custom menu fields
		require_once( get_theme_file_path( '/lib/menus/menu-item-custom-fields.php' ) );
				
		// Load page header
		require_once( get_theme_file_path( '/lib/sections/single/page-header.php' ) );
		
		// Load video header
		require_once( get_theme_file_path( '/lib/sections/single/video-header.php' ) );
			
		// Load page title
		require_once( get_theme_file_path( '/lib/sections/single/page-titles.php' ) );
			
		// Load login functions
		require_once( get_theme_file_path( '/lib/inc/login-functions.php' ) );
						
		// Load BuddyPress functions
		if ( function_exists( 'bp_is_active' ) ) {
			require_once( get_theme_file_path( '/lib/inc/buddypress-functions.php' ) );
		}

		// Load bbPress functions
		if ( function_exists( 'is_bbpress' ) ) {
			require_once( get_theme_file_path( '/lib/inc/bbpress-functions.php' ) );
		}
		
		// Load Woocommerce functions
		if ( function_exists( 'is_woocommerce' ) ) {
			require_once( get_theme_file_path( '/lib/inc/woocommerce-functions.php' ) );
		}
		
		// Load Paid Membership Pro functions
		if ( defined( 'PMPRO_VERSION' ) ) {
			require_once( get_theme_file_path( '/lib/inc/pmpro-functions.php' ) );
		}

		// Load Elementor functions
		if ( defined( 'ELEMENTOR_VERSION' ) ) {
			require_once( get_theme_file_path( '/lib/inc/elementor-functions.php' ) );
		}
									
	}
}
add_action( 'after_setup_theme', 'ghostpool_load_theme_functions' );
				
/**
 * Registered image sizes
 *
 */
if ( ! function_exists( 'ghostpool_image_sizes' ) ) {
	function ghostpool_image_sizes() {				
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'gp_small_image', 75, 75, true );
		add_image_size( 'gp_list_image', 250, 135, true );
		add_image_size( 'gp_square_image', 300, 300, true );
		add_image_size( 'gp_featured_image', 864, 467, true );
		add_image_size( 'gp_column_image', 727, 393, true );
		add_image_size( 'gp_related_image', 414, 224, true );
		add_image_size( 'gp_featured_box_small_image', 330, 240, true );	
		add_image_size( 'gp_featured_box_large_image', 600, 480, true );	
		add_image_size( 'gp_featured_box_full_image', 1260, 480, true );		
	}
}
add_action( 'after_setup_theme', 'ghostpool_image_sizes' );

/**
 * Sets the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 *
 */
function ghostpool_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'ghostpool_sidebar_content_width', 1140 );
}
add_action( 'after_setup_theme', 'ghostpool_content_width', 0 );
		
/**
 * Enqueues scripts and styles.
 *
 */	
if ( ! function_exists( 'ghostpool_scripts' ) ) {
	function ghostpool_scripts() {
	
		global $wp_query;

		wp_enqueue_style( 'ghostpool-style', get_template_directory_uri() . '/style.css', array(), AARDVARK_THEME_VERSION );
		
		wp_style_add_data( 'ghostpool-style', 'rtl', 'replace' );

		if ( 'enabled' === ghostpool_option( 'fontawesome5' ) ) {

			wp_deregister_style( 'font-awesome' );
			wp_enqueue_style( 'fontawesome5', get_template_directory_uri() . '/lib/fonts/fontawesome5/css/all.min.css', array(), AARDVARK_THEME_VERSION );
		
		} else {
				
			if ( file_exists( plugin_dir_url( 'js_composer' ) . 'assets/lib/bower/font-awesome/css/font-awesome.min.css' ) ) {
				$font_url = plugin_dir_url( 'js_composer' ) . 'assets/lib/bower/font-awesome/css/font-awesome.min.css';
			} else {
				$font_url = get_template_directory_uri() . '/lib/fonts/font-awesome/css/font-awesome.min.css';
			}		
			wp_enqueue_style( 'font-awesome', $font_url, array(), AARDVARK_THEME_VERSION );
		
		}
						
		if ( ghostpool_option( 'lightbox' ) != 'disabled' ) {
			wp_enqueue_style( 'featherlight', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.min.css', array(), AARDVARK_THEME_VERSION );
			wp_enqueue_style( 'featherlight-gallery', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.gallery.min.css', array(), AARDVARK_THEME_VERSION );
		}
			
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) { 
			wp_enqueue_script( 'comment-reply' );
		}
					
		wp_enqueue_script( 'imagesloaded' );
		
		wp_register_script( 'particles-js', get_template_directory_uri() . '/lib/scripts/particles.min.js', false, AARDVARK_THEME_VERSION, true );

		wp_enqueue_script( 'placeholder', get_template_directory_uri() . '/lib/scripts/placeholders.min.js', false, AARDVARK_THEME_VERSION, true );
									
		if ( ghostpool_option( 'lightbox' ) != 'disabled' ) {
			wp_enqueue_script( 'featherlight', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.min.js', array( 'jquery' ), AARDVARK_THEME_VERSION, true );
			wp_enqueue_script( 'featherlight-gallery', get_template_directory_uri() . '/lib/scripts/featherlight/featherlight.gallery.min.js', array( 'jquery' ), AARDVARK_THEME_VERSION, true );
		}
			
		if ( ghostpool_option( 'back_to_top' ) != 'gp-no-back-to-top' ) { 
			wp_enqueue_script( 'jquery-totop', get_template_directory_uri() . '/lib/scripts/jquery.ui.totop.min.js', array( 'jquery' ), AARDVARK_THEME_VERSION, true );
		}	

		wp_enqueue_script( 'jquery-flexslider', get_template_directory_uri() . '/lib/scripts/jquery.flexslider-min.js', array( 'jquery' ), AARDVARK_THEME_VERSION, true );
		
		wp_enqueue_script( 'isotope', get_template_directory_uri() . '/lib/scripts/isotope.pkgd.min.js', false, AARDVARK_THEME_VERSION, true );

		wp_enqueue_script( 'lazyload', get_template_directory_uri() . '/lib/scripts/jquery.lazyload.min.js', array( 'jquery' ), AARDVARK_THEME_VERSION, true );
		
		wp_enqueue_script( 'jquery-infinitescroll', get_template_directory_uri() . '/lib/scripts/jquery.infinitescroll.min.js', array( 'jquery' ), AARDVARK_THEME_VERSION, true );
				
		wp_enqueue_script( 'ghostpool-custom', get_template_directory_uri() . '/lib/scripts/custom.js', array( 'jquery' ), AARDVARK_THEME_VERSION, true );
		
		if ( is_ssl() ) {
			$url = esc_url( 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] );
		} else { 
			$url = esc_url( 'http://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'] );
		}
			
		wp_localize_script( 'ghostpool-custom', 'ghostpool_script', array(
			'url'	   					   => $url,
			'max_num_pages' 			   => $wp_query->max_num_pages,
			'get_template_directory_uri'   => get_template_directory_uri(),
			'bp_item_tabs_nav_text'		   => esc_html__( 'Navigation', 'aardvark' ),
			'hide_move_primary_menu_links' => ghostpool_option( 'hide_move_primary_menu_links' ),
			'scroll_to_fixed_header'       => ghostpool_option( 'scroll_to_fixed_header', 'height' ),
			'lightbox' 					   => ghostpool_option( 'lightbox' ),
			'automatic_video_resizing' 	   => ghostpool_option( 'automatic_video_resizing' ),
		) );
						
	}
}
add_action( 'wp_enqueue_scripts', 'ghostpool_scripts' );

/**
 * Adds custom classes to the array of body classes.
 *
 */
if ( ! function_exists( 'ghostpool_body_classes' ) ) {
	function ghostpool_body_classes( $classes ) {
	
		// Get settings from query_var
		$settings = get_query_var( 'ghostpool_page_settings' );
		if ( $settings ) {
			extract( $settings );
		}
				
		$classes[] = 'gp-theme';
		$classes[] = ghostpool_option( 'theme_layout' );
		$classes[] = isset( $layout ) ? $layout : '';		
		$classes[] = isset( $header ) ? $header : '';
		$classes[] = ghostpool_option( 'sidebar_display' );
		$classes[] = ghostpool_option( 'back_to_top' );
		$bp_header_layout = ghostpool_option( 'bp_header_layout' ) != 'default' ? ghostpool_option( 'bp_header_layout' ) : ghostpool_option( 'header_layout' );
		if ( function_exists( 'bp_is_active' ) && bp_is_activity_component() ) {
			$classes[] =  ghostpool_option( 'bp_activity_header_layout' ) != 'default' ? ghostpool_option( 'bp_activity_header_layout' ) : $bp_header_layout;
		} elseif ( function_exists( 'bp_is_active' ) && bp_is_members_component() ) {
			$classes[] =  ghostpool_option( 'bp_members_header_layout' ) != 'default' ? ghostpool_option( 'bp_members_header_layout' ) : $bp_header_layout;
		} elseif ( function_exists( 'bp_is_active' ) && bp_is_groups_component() ) {
			$classes[] = ghostpool_option( 'bp_groups_header_layout' ) != 'default' ? ghostpool_option( 'bp_groups_header_layout' ) : $bp_header_layout;
		} elseif ( function_exists( 'bp_is_active' ) && bp_is_register_page() ) {
			$classes[] = ghostpool_option( 'bp_register_header_layout' ) != 'default' ? ghostpool_option( 'bp_register_header_layout' ) : $bp_header_layout;	
		} else {
			$classes[] = get_post_meta( get_the_ID(), 'gp_page_header_layout', true ) && get_post_meta( get_the_ID(), 'gp_page_header_layout', true ) != 'default' ? get_post_meta( get_the_ID(), 'gp_page_header_layout', true ) : ghostpool_option( 'header_layout' );
		}
		$classes[] = get_post_meta( get_the_ID(), 'gp_page_header_display', true ) && get_post_meta( get_the_ID(), 'gp_page_header_display', true ) != 'default' ? get_post_meta( get_the_ID(), 'gp_page_header_display', true ) : ghostpool_option( 'header_display' );
		$classes[] = ghostpool_option( 'header_format' );
		$classes[] = ghostpool_option( 'fixed_header' );
		$classes[] = ghostpool_option( 'fixed_header_type' );
		$classes[] = ghostpool_option( 'top_header' );
		$classes[] = ghostpool_option( 'cart_button' );
		$classes[] = ghostpool_option( 'search_button' );
		$classes[] = ghostpool_option( 'profile_button' );
		$classes[] = ghostpool_option( 'footer_sticky' );
		
		if ( is_page_template( 'homepage-template.php' ) ) {
			$classes[] = 'gp-homepage';
		}
		
		if ( defined( 'TSS_VERSION' ) ) {	
			$classes[] = 'gp-sticky-sidebars';	
		}	
		
		return $classes;
		
	}
}
add_filter( 'body_class', 'ghostpool_body_classes' );

/**
 * Register sidebars
 *
 */
if ( ! function_exists( 'ghostpool_widgets_init' ) ) {
	function ghostpool_widgets_init() {

		register_sidebar( array( 
			'name'          => esc_html__( 'Right Sidebar', 'aardvark' ),
			'id'            => 'gp-right-sidebar',
			'description'   => esc_html__( 'Displayed on posts, pages and post categories.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) );

		register_sidebar( array( 
			'name'          => esc_html__( 'Left Sidebar', 'aardvark' ),
			'id'            => 'gp-left-sidebar',
			'description'   => esc_html__( 'Displayed on posts, pages and post categories.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) ); 

		register_sidebar( array(
			'name'          => esc_html__( 'Side Menu', 'aardvark' ),
			'id'            => 'gp-side-menu',
			'description'   => esc_html__( 'Displayed in the side menu.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) );
			
		register_sidebar( array(
			'name'          => esc_html__( 'Footer 1', 'aardvark' ),
			'id'            => 'gp-footer-1',
			'description'   => esc_html__( 'Displayed as the first column in the footer.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) );        

		register_sidebar( array(
			'name'          => esc_html__( 'Footer 2', 'aardvark' ),
			'id'            => 'gp-footer-2',
			'description'   => esc_html__( 'Displayed as the second column in the footer.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) );        
	
		register_sidebar( array(
			'name'          => esc_html__( 'Footer 3', 'aardvark' ),
			'id'            => 'gp-footer-3',
			'description'   => esc_html__( 'Displayed as the third column in the footer.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) );        
	
		register_sidebar( array(
			'name'          => esc_html__( 'Footer 4', 'aardvark' ),
			'id'            => 'gp-footer-4',
			'description'   => esc_html__( 'Displayed as the fourth column in the footer.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) );      

		register_sidebar( array(
			'name'          => esc_html__( 'Footer 5', 'aardvark' ),
			'id'            => 'gp-footer-5',
			'description'   => esc_html__( 'Displayed as the fifth column in the footer.', 'aardvark' ),
			'before_widget' => '<div id="%1$s" class="widget widget-content %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title widgettitle">',
			'after_title'   => '</h3>',
		) ); 
			
	}
}
add_action( 'widgets_init', 'ghostpool_widgets_init' );
		
/**
 * Load theme settings
 *
 */	
if ( ! function_exists( 'ghostpool_load_settings_in_header' ) ) {
	function ghostpool_load_settings_in_header() {

		// Run function
		ghostpool_load_settings();

	}	
}
add_action( 'wp_head', 'ghostpool_load_settings_in_header', 151 );
	
/**
 * Add custom content to header
 *
 */	
if ( ! function_exists( 'ghostpool_custom_header_content' ) ) {
	function ghostpool_custom_header_content() {
	
		// Title fallback for versions earlier than WordPress 4.1
		if ( ! function_exists( '_wp_render_title_tag' ) && ! function_exists( 'ghostpool_render_title' ) ) {
			function ghostpool_render_title() { ?>
				<title><?php wp_title( '|', true, 'right' ); ?></title>
			<?php }
		}

		// Add custom header code
		if ( ghostpool_option( 'custom_header_code' ) ) {
			echo ghostpool_option( 'custom_header_code' );
		}
		
		// Add custom JavaScript code
		if ( ghostpool_option( 'js_code' ) ) {
			if ( strpos( ghostpool_option( 'js_code' ), '<script ' ) !== false ) { 
				echo ghostpool_option( 'js_code' ); 
			} else {
				$js_code = str_replace( array( '<script>', '</script>' ), '', ghostpool_option( 'js_code' ) );
				echo '<script>' . $js_code . '</script>';
			}    
		}

	}	
}
add_action( 'wp_head', 'ghostpool_custom_header_content' );
	
/**
 * Content added to footer
 * Previously called ghostpool_rich_snippets
 */	
if ( ! function_exists( 'ghostpool_custom_footer_content' ) ) {
	function ghostpool_custom_footer_content() {

		// Add custom footer code
		if ( ghostpool_option( 'custom_footer_code' ) ) {
			echo ghostpool_option( 'custom_footer_code' );
		}

		// Get title
		if ( get_post_meta( get_the_ID(), 'gp_custom_title', true ) ) { 
			$title = get_post_meta( get_the_ID(), 'gp_custom_title', true );
		} else {
			$title = get_the_title( get_the_ID() );
		}
		
		// Get featured image dimensions
		$image = ghostpool_image_data( ghostpool_option( 'post_image_size' ) );

		if ( is_single() ) {
		
			global $post;

			echo '<script type="application/ld+json">';
		
			echo '{
			"@context": "http://schema.org/",
				"@type": "Article",
				"mainEntityOfPage": {
					"@type": "WebPage",
					"url": "' . esc_url( get_permalink( get_the_ID() ) ) . '"
				},	
				"headline": "' . esc_attr( $title ) . '",	
				"author": "' . get_the_author_meta( 'display_name', $post->post_author ) . '",	
				"datePublished": "' . get_the_time( 'Y-m-d' ) . '",	
				"dateModified": "' . get_the_modified_date( 'Y-m-d' ) . '",		
				"image": {
					"@type": "ImageObject",
					"url": "' . esc_url( wp_get_attachment_url( get_post_thumbnail_id( get_the_ID() ) ) ) . '",
					"width": "' . absint( $image['width'] ) . '",
					"height": "' . absint( $image['height'] ) . '"
				},
				"publisher": {
					"@type": "Organization",
					"url": "' . esc_url( get_permalink( get_the_ID() ) ) . '",
					"name": "' . get_bloginfo( 'name' ) . '",
					"logo": {
						"@type": "ImageObject",
						"url": "' . esc_url( ghostpool_image_url( ghostpool_option( 'logo', 'standard' ) ) ) . '",
						"width": "' . absint( ghostpool_option( 'logo', 'width' ) ) . '",
						"height": "' . absint( ghostpool_option( 'logo', 'height' ) ) . '"
					}
				},
				"text": "' . wp_strip_all_tags( get_the_excerpt( get_the_ID() ), true ) . '"
				}';
			
			echo '</script>';

		}
				
	}
}
add_action( 'wp_footer', 'ghostpool_custom_footer_content' );
					
/**
 * Navigation user meta
 *
 */	
if ( ! function_exists( 'ghostpool_nav_user_meta' ) ) {
	function ghostpool_nav_user_meta( $user_id = NULL ) {
	
		// So this can be used without hooking into user_register
		if ( ! $user_id ) {
			$user_id = get_current_user_id(); 
		}

		// Set the default properties if it has not been set yet
		if ( ! get_user_meta( $user_id, 'managenav-menuscolumnshidden', true ) ) {
			$meta_value = array( 'link-target', 'xfn', 'description' );
			update_user_meta( $user_id, 'managenav-menuscolumnshidden', $meta_value );
		}
	
	}	
}
add_action( 'admin_init', 'ghostpool_nav_user_meta' );

/**
 * Insert breadcrumbs
 *
 */	
if ( ! function_exists( 'ghostpool_breadcrumbs' ) ) {
	function ghostpool_breadcrumbs() {
		if ( function_exists( 'yoast_breadcrumb' ) && ! is_front_page() ) { 
			$breadcrumbs = yoast_breadcrumb( '<div id="gp-breadcrumbs" class="gp-active">', '</div>' );
		} else {
			$breadcrumbs = '';
		}
		return $breadcrumbs;
	}
}

/**
 * Header advertisement
 *
 */	
function ghostpool_add_header_advertisement() {
	if ( function_exists( 'the_ad_placement' ) ) { the_ad_placement( 'header' ); }
}
add_action( 'ghostpool_begin_content_wrapper', 'ghostpool_add_header_advertisement' );

/**
 * Footer advertisement
 *
 */	
function ghostpool_add_footer_advertisement() {
	if ( function_exists( 'the_ad_placement' ) ) { the_ad_placement( 'footer' ); }
}
add_action( 'ghostpool_end_content_wrapper', 'ghostpool_add_footer_advertisement' );

/**
 * Get author name
 *
 */	
if ( ! function_exists( 'ghostpool_author_name' ) ) {
	function ghostpool_author_name( $post_id = '', $author_link = true ) {
		global $post;
		if ( get_post_meta( $post_id, 'ghostpool_post_submission_username', true ) ) {	
			$author = get_post_meta( $post_id, 'ghostpool_post_submission_username', true );
		} elseif ( $author_link == true ) {
			$author = apply_filters( 'ghostpool_author_url', '<a href="' . esc_url( get_author_posts_url( $post->post_author ) ) . '">' . get_the_author_meta( 'display_name', $post->post_author ) . '</a>', $post );
		} else {
			$author = get_the_author_meta( 'display_name', $post->post_author );
		}
		return apply_filters( 'ghostpool_author_url', $author, $post );
	}
}

/**
 * Change password protect text
 *
 */	
if ( ! function_exists( 'ghostpool_password_form' ) ) {
	function ghostpool_password_form() {
		global $post;
		$label = 'pwbox-' . ( empty( $post->ID ) ? rand() : $post->ID );
		$o = '<form action="' . esc_url( site_url( 'wp-login.php?action=postpass', 'login_post' ) ) . '" method="post">
		<p>' . esc_html__( 'To view this protected post, enter the password below:', 'aardvark' ) . '</p>
		<label for="' . $label . '"><input name="post_password" id="' . $label . '" type="password" size="20" maxlength="20" /></label> <input type="submit" class="pwsubmit" name="Submit" value="' .  esc_attr__( 'Submit', 'aardvark' ) . '" />
		</form>
		';
		return $o;
	}
}
add_filter( 'the_password_form', 'ghostpool_password_form' );

/**
 * Redirect empty search to search page
 *
 */	
if ( ! function_exists( 'ghostpool_empty_search' ) ) {
	function ghostpool_empty_search( $query ) {
		global $wp_query;
		if ( isset( $_GET['s'] ) && ( $_GET['s'] == '' ) ) {
			$wp_query->set( 's', ' ' );
			$wp_query->is_search = true;
		}
		return $query;
	}
}
add_action( 'pre_get_posts', 'ghostpool_empty_search' );

/**
 * Add lightbox class to image links
 *
 */
if ( ! function_exists( 'ghostpool_lightbox_image_link' ) ) {
	function ghostpool_lightbox_image_link( $content ) {
		if ( ghostpool_option( 'lightbox' ) != 'disabled' ) {
			if ( ghostpool_option( 'lightbox' ) == 'group_images' ) {
				$lightbox = ' data-lightbox="gallery" ';
			} else {
				$lightbox = ' data-featherlight="image" ';
			}
			$pattern = "/<a(.*?)href=('|\")(.*?).(jpg|jpeg|png|gif|bmp|ico)('|\")(.*?)>/i";
			preg_match_all( $pattern, $content, $matches, PREG_SET_ORDER );
			foreach ( $matches as $val ) {
				$pattern = '<a' . $val[1] . 'href=' . $val[2] . $val[3] . '.' . $val[4] . $val[5] . $val[6] . '>';
				$replacement = '<a' . $val[1] . 'href=' . $val[2] . $val[3] . '.' . $val[4] . $val[5] . $lightbox . $val[6] . '>';
				$content = str_replace( $pattern, $replacement, $content );			
			}
			return $content;
		} else {
			return $content;
		}
	}	
}
add_filter( 'the_content', 'ghostpool_lightbox_image_link' );	
add_filter( 'wp_get_attachment_link', 'ghostpool_lightbox_image_link' );
add_filter( 'bbp_get_reply_content', 'ghostpool_lightbox_image_link' );

/**
 * Advanced ads background support
 *
 */
if ( ! function_exists( 'ghostpool_background_ad_selector' ) ) {
	function ghostpool_background_ad_selector() {
		return '#gp-page-wrapper';
	}
}
add_filter( 'advanced-ads-pro-background-selector', 'ghostpool_background_ad_selector' );

/**
 * Get image data
 *
 */
if ( ! function_exists( 'ghostpool_image_data' ) ) {
	function ghostpool_image_data( $dimensions, $format = '' ) {
		
		$dimensions = str_replace( ' ', '', $dimensions );
		$matches = null;
		
		if ( preg_match( '/(\d+)x(\d+)/', $dimensions, $matches ) ) {
		
			$dimensions = array( 
				$matches[1],
				$matches[2],
			);
			
			$image_width = $matches[1];
			$image_height = $matches[2];			
			
		} elseif ( '' === $dimensions ) {
		
			global $_wp_additional_image_sizes;
		
			if ( 'gp-posts-list' === $format ) {
				$dimensions = 'gp_list_image';
			} elseif ( 'gp-posts-large' === $format ) {
				$dimensions = 'gp_featured_image';
			} else {
				$dimensions = 'gp_column_image';
			}

			$image_width = $_wp_additional_image_sizes[ $dimensions ]['width'];
			$image_height = $_wp_additional_image_sizes[ $dimensions ]['height'];
					
		} else {

			global $_wp_additional_image_sizes;
			
			if ( in_array( $dimensions, array( 'thumbnail', 'medium', 'medium_large', 'large' ) ) ) {
				$image_width = get_option( "{$dimensions}_size_w" );
				$image_height = get_option( "{$dimensions}_size_h" );			
			} elseif ( isset( $_wp_additional_image_sizes[ $dimensions ] ) ) {
				$image_width = $_wp_additional_image_sizes[ $dimensions ]['width'];
				$image_height = $_wp_additional_image_sizes[ $dimensions ]['height'];
			} else {
				$dimensions = 'thumbnail';
				$image_width = get_option( "{$dimensions}_size_w" );
				$image_height = get_option( "{$dimensions}_size_h" );		
			}

		}	
					
		return array(
			'name' => $dimensions,
			'width' => $image_width,
			'height' => $image_height,
		);
			
	}
}

/**
 * Add units to values if left empty
 *
 */
if ( ! function_exists( 'ghostpool_add_units' ) ) {
	function ghostpool_add_units( $value = '', $units = 'px' ) {
		if ( is_numeric( $value ) ) {
			$value = $value . $units;
		}
		return $value;	
	}
}

/**
 * Get image URL
 *
 */
if ( ! function_exists( 'ghostpool_image_url' ) ) {
	function ghostpool_image_url( $value ) {
		if ( is_numeric( $value ) ) {
			$value = wp_get_attachment_url( $value );
		}
		return $value;	
	}
}